/************************************************************************
 *
 * \file: AndroidAutoDeviceMonitor.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Demo application
 *
 * \author: J. Harder / ADITG/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <adit_logging.h>
#include <aauto/AautoLogging.h>
#include "session/Session.h"
#include "Server.h"
#include "ServerMonitor.h"
#include "AutoSmoketest.h"
#include "libtestbedapp.h"
#include "Testbed.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

using namespace uspi;

void ServerMonitor::onDeviceFound(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    // any AOAP supporting device

    DeviceInfo ddInfo = inUsbDevice->getInfo();

    LOG_INFO((demo, "AOAP device found: poduct=%s",
              ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str() ));

    // switch done by the FeatureDiscovery
    struct SpiAoapInformation tmpParam;
    tmpParam.manufacturer   = "Android";
    tmpParam.modelName      = "Android Auto";
    tmpParam.description    = "Android Auto";
    tmpParam.version        = "1.3";
    tmpParam.uri            = "http://www.android.com/auto";
    tmpParam.serial         = "000000001234567";
    tmpParam.enableAudio    = 0;

    if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
    {
        if (AutoSmoketest::instance().getTestError() == NODEVICE)
        {
            /* first device which supports AOAP --> continue */
        }
        else
        {
            std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> deviceFound = AutoSmoketest::instance().getTestDevice();
            /* operator bool() - return false if the shared_ptr is a null pointer */
            if (true == deviceFound.operator bool())
            {
                LOG_INFO((demo,"AST: Cannot test device %s because test already in progress for device %s.",
                        to_string(ddInfo).c_str(), deviceFound->getInfo().getDeviceInfo(DSYS_SERIAL).c_str()));
                /* test in progress for another device --> test will be not executed for newly device */
                return;
            }
            else
            {
                if (AutoSmoketest::instance().getTestError() == DEVICEALREADYINAOAP)
                {
                    LOG_INFO((demo,"AST: Device list is empty, but state = %s - use this device %s for testing",
                            AutoSmoketest::instance().getErrorString().c_str(), to_string(ddInfo).c_str()));
                }
                else
                {
                    LOG_WARN((demo,"AST: Test in progress (%s), but no device in listed.",
                            AutoSmoketest::instance().getErrorString().c_str()));

                    /* test in progress, but we listed no device --> don't test and let the AST fail */
                    return;
                }
            }

        }
    }

    if (DiscoveryError::OK != inUsbDevice->switchDevice(FD_PROTOCOL_GOOGLE_AOAP, &tmpParam, 1000))
    {
        LOG_ERROR((demo, "switchDevice() failed."));
    }
    else
    {
        LOG_INFO((demo, "switchDevice() success."));
        if(Testbed::instance().getTestbedMode())
        {
            Testbed::instance().setDeviceState(SWITCHED);
        }

        if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
        {
            LOG_INFO((demo,"AST: store device %s into list", to_string(ddInfo).c_str()));
            AutoSmoketest::instance().addDevice(inUsbDevice);

            AutoSmoketest::instance().setTestError(DEVICENOTSWITCHED);
        }

        // remove if already stored/
        auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
        if (found != transportInfos.end()) {
            transportInfos.erase(found);
        }

        // insert empty information to indicate that the switch was triggered by us
        aoapTransportInfo_t info = {0,0};
        transportInfos.insert(std::pair<std::string, aoapTransportInfo_t>(ddInfo.getDeviceInfo(DSYS_SERIAL), info));
    }
}

void ServerMonitor::onDeviceSwitched(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    // set location data which use for sensor source.
    TestParameter testPara;
    testPara.setLocation(getLocation());

    pthread_mutex_lock(&Server::instance().sessionMutex);   // session mutex lock 

    DeviceInfo ddInfo = inUsbDevice->getInfo();

    // any device in accessory mode, either switched by us or already in accessory mode
    LOG_INFO((demo, "AOAP device found in AOAP mode product=%s %s",
            ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str(), ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));


    auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));

    // check if transport info already exist for this device
    if (found == transportInfos.end())
    {
        // not found
        LOG_WARN((demo, "accessory mode in device %s %s without previous switch connected",
                ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str(), ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));

        // Reset the device only if in Automatic Smoketest mode
        if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
        {
            std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> deviceFound = AutoSmoketest::instance().getTestDevice(inUsbDevice);
            /* operator bool() - return false if the shared_ptr is a null pointer */
            if (true == deviceFound.operator bool())
            {
                LOG_INFO((demo,"AST: Device %s is known, but in AOAP mode (state=%s).",
                        deviceFound->getInfo().getDeviceInfo(DSYS_SERIAL).c_str(), AutoSmoketest::instance().getErrorString().c_str() ));
            }
            else
            {
                LOG_INFO((demo,"AST: Device %s is unknown, but already in AOAP mode.", to_string(ddInfo).c_str() ));

                AutoSmoketest::instance().setTestError(DEVICEALREADYINAOAP);
            }
            LOG_INFO((demo,"AST: Reset device %s.", to_string(ddInfo).c_str() ));
            resetDevice(inUsbDevice, true);
        }
        else
        {
            // switch device and retrieve info
            aoapTransportInfo_t info = {0,0};
            bool switchResult = Server::instance().switchDevice(ddInfo.getiDeviceInfo(DSYS_IDVENDOR), \
                                                                ddInfo.getiDeviceInfo(DSYS_IDPRODUCT), \
                                                                ddInfo.getDeviceInfo(DSYS_SERIAL), &info);
            if (true != switchResult)
            {
                // TODO error handling
                LOG_ERROR((demo, "switchDevice() failed."));
            }
            else
            {
                LOG_INFO((demo, "switchDevice() success."));

                // remove if already stored
                // TODO do differently!
                auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
                if (found != transportInfos.end())
                {
                    transportInfos.erase(found);
                }

                transportInfos.insert(std::pair<std::string, aoapTransportInfo_t>(ddInfo.getDeviceInfo(DSYS_SERIAL), info));

                LOG_INFO((demo, "re-create AndroidAuto session %s", ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));

                auto session = Server::instance().createSession(ddInfo.getDeviceInfo(DSYS_SERIAL), info, testPara);
                // info is not guaranteed to live longer
                // TODO error handling

                // transport info no longer required
                found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
                if (found != transportInfos.end())
                {
                    transportInfos.erase(found);
                }
            }
        }
    }
    else
    {
        // transport info already exist

        //save device for reset later in AutoSmoketest
        if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
        {
            std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> deviceFound = AutoSmoketest::instance().getTestDevice(inUsbDevice);
            /* operator bool() - return false if the shared_ptr is a null pointer */
            if (true == deviceFound.operator bool())
            {
                DeviceInfo ddFoundInfo = deviceFound->getInfo();
                LOG_INFO((demo,"AST: remove device %s from list", to_string(ddFoundInfo).c_str() ));
                AutoSmoketest::instance().removeDevice();
            }

            LOG_INFO((demo,"AST: store device %s into list", to_string(ddInfo).c_str()));
            AutoSmoketest::instance().addDevice(inUsbDevice);

            AutoSmoketest::instance().setTestError(DEVICESWITCHED);
        }

        // switch device to retrieve AOAP transport information
        aoapTransportInfo_t info = {0,0};
        bool switchResult = Server::instance().switchDevice(ddInfo.getiDeviceInfo(DSYS_IDVENDOR), \
                                                            ddInfo.getiDeviceInfo(DSYS_IDPRODUCT), \
                                                            ddInfo.getDeviceInfo(DSYS_SERIAL), &info);
        if (true != switchResult)
        {
            // TODO error handling
            LOG_ERROR((demo, "switchDevice() failed."));
        }
        else
        {
            LOG_INFO((demo, "switchDevice() success."));

            LOG_INFO((demo, "create AndroidAuto session %s", ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));
            auto session = Server::instance().createSession(ddInfo.getDeviceInfo(DSYS_SERIAL), info, testPara);
        }
        // transport info no longer required
        transportInfos.erase(found);
    }

    pthread_mutex_unlock(&Server::instance().sessionMutex);   // session mutex unlock
}

void ServerMonitor::onDeviceLost(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    DeviceInfo ddInfo = inUsbDevice->getInfo();

      //todo: pop device from list when removed (this is causing problem:segfault)
//    if(AutoSmoketest::instance().getTestMode()==AUTOMATIC)
//    {
//        auto& usbDevList = AutoSmoketest::instance().getUsbDevInfoList();
//        printf("removing device from %d %d %s reset list", inDevice.vendorId, inDevice.productId,inDevice.sysPath.c_str());
//        usbDevList.pop_back();
//    }

    // any lost device
    LOG_INFO((demo, "AOAP device lost"));

    if(Testbed::instance().getTestbedMode())
    {
        if(Testbed::instance().getDeviceState() != SWITCHED)
        {
            Testbed::instance().setDeviceState(DISCONNECTED);
            if(Testbed::instance().getCurrentAppState() == TBA_STATE_BACKGROUND)
            {
                Testbed::instance().controlSurface();
                Testbed::instance().adit_testbed_lost();
                Testbed::instance().setDeviceState(INIT);
            }
        }
        else
        {
            Testbed::instance().setDeviceState(INIT);
        }
    }

    pthread_mutex_lock(&Server::instance().sessionMutex);   // session mutex lock

    // lost device is not necessarily one with a session
    auto session = Server::instance().getSession(ddInfo.getDeviceInfo(DSYS_SERIAL));
    if (session != nullptr)
    {
        LOG_INFO((demo, "destroy AndroidAuto session %s", ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));
        Server::instance().destroySession(ddInfo.getDeviceInfo(DSYS_SERIAL));
    }
    pthread_mutex_unlock(&Server::instance().sessionMutex);   // session mutex unlock
}

void ServerMonitor::onDeviceChanged(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    DeviceInfo ddInfo = inUsbDevice->getInfo();
    LOG_INFO((demo, "device %s changed", ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));
}

void ServerMonitor::onDeviceError(adit::uspi::DiscoveryError inErrorCode)
{
    LOG_INFO((demo, "errorCb triggered with error %s", to_str(inErrorCode)));
}


} } // namespace adit { namespace aauto {
